// (c) 1999 - 2025 OneSpan North America Inc. All rights reserved.


/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import MSSOrchestration

class LocalTransactionViewController: UIViewController {
    @IBOutlet weak var userIdLabel: UILabel!
    @IBOutlet weak var userId: UILabel!
    @IBOutlet weak var beneficiaryLabel: UILabel!
    @IBOutlet weak var beneficiary: UITextField!
    @IBOutlet weak var beneficiaryError: UILabel!
    @IBOutlet weak var ibanLabel: UILabel!
    @IBOutlet weak var iban: UITextField!
    @IBOutlet weak var ibanError: UILabel!
    @IBOutlet weak var amountLabel: UILabel!
    @IBOutlet weak var amount: UITextField!
    @IBOutlet weak var amountError: UILabel!
    @IBOutlet weak var protectionTitle: UILabel!
    @IBOutlet weak var noPasswordButton: UIButton!
    @IBOutlet weak var passwordButton: UIButton!
    @IBOutlet weak var biometricButton: UIButton!
    @IBOutlet weak var signTransactionButton: UIButton!
    @IBOutlet weak var scroll: UIScrollView!
    
    private let orchestrationDelegate = OrchestrationSampleDelegate()
    private let userAuthenticationDelegate = UserAuthenticationViewController()

    private var protectionSelected: UIButton?
    private var responder: UIView?
    private var progressDialog: UIView?
    private var orchestrator: Orchestrator?

    override func viewDidLoad() {
        super.viewDidLoad()
        setupView()
        setupOrchestration()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(keyboardWillShow(_:)),
                                               name: UIResponder.keyboardWillShowNotification,
                                               object: nil)
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(keyboardWillHide(notification:)),
                                               name: UIResponder.keyboardWillHideNotification,
                                               object: nil)
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        dismissKeyboard()
        NotificationCenter.default.removeObserver(self)
    }
    
    // MARK: Setup
    private func setupView() {
        title = "title_activity_local_transaction".localized
        navigationItem.hidesBackButton = false
        
        userIdLabel.text = "prompt_user_id".localized
        beneficiaryLabel.text = "prompt_beneficiary".localized
        ibanLabel.text = "prompt_iban".localized
        amountLabel.text = "prompt_amount".localized
        
        protectionTitle.text = "protection_title".localized
        noPasswordButton.setTitle("protection_type_no_password".localized, for: UIControl.State.normal)
        passwordButton.setTitle("protection_type_password".localized, for: UIControl.State.normal)
        biometricButton.setTitle("protection_type_biometric".localized, for: UIControl.State.normal)
        
        signTransactionButton.setTitle("btn_sign_transaction".localized, for: UIControl.State.normal)
        
        navigationController?.navigationItem.accessibilityLabel = "backButton"
        
        // initialize buttons states
        protectionSelected = noPasswordButton
        protectionSelected?.isSelected = true
        
        // Scroll and form configuration
        let gesture = UITapGestureRecognizer(target: self, action: #selector(dismissKeyboard))
        scroll.addGestureRecognizer(gesture)
        
        signTransactionButton.layer.cornerRadius = 3
        beneficiary.returnKeyType = UIReturnKeyType.next
        iban.returnKeyType = UIReturnKeyType.next
        
        userId.text = SharedPreferenceStorage.getActivatedUser()
        
        beneficiary.delegate = self
        iban.delegate = self
        amount.delegate = self
    }
    
    private func setupOrchestration() {
        orchestrationDelegate.progressDialog = progressDialog
        orchestrationDelegate.viewController = self
        orchestrator = OrchestratorUtils.getOrchestrator(delegate: orchestrationDelegate)
                
        // Used for custom password instead of default one
        guard let orchestrator = orchestrator else {
            assertionFailure("orchestrator is nil in LocalTransactionViewController")
            return
        }
        
        userAuthenticationDelegate.useExternalPassword(orchestrator: orchestrator, viewController: self)
    }
    
    // MARK: IBActions
    @IBAction func onProtectionButtonClick(_ sender: Any) {
        protectionSelected?.isSelected = false
        protectionSelected = sender as? UIButton
        protectionSelected?.isSelected = true
    }
    
    @IBAction func onSignTransactionButtonClick(_ sender: Any) {
        guard let iban = self.iban.text,
              let amount = self.amount.text,
              let beneficiary = self.beneficiary.text,
              let identifier = userId.text
        else {
            assertionFailure("at least one of the following is nil: iban,amount, beneficiary, userId")
            return
        }
        beneficiaryError.text = ""
        ibanError.text = ""
        amountError.text = ""
        
        // Get beneficiary
        if !beneficiary.isEmpty {
            let beneficiaryPattern = "[a-zA-Z0-9]{1,16}"
            let beneficiaryMatcher = NSPredicate(format: "SELF MATCHES %@", beneficiaryPattern)
            if !beneficiaryMatcher.evaluate(with: beneficiary) {
                beneficiaryError.text = "error_local_transaction_beneficiary_invalid".localized
                return
            }
        }
        
        // Get IBAN
        
        if !iban.isEmpty {
            let ibanPattern = "[a-zA-Z0-9]{1,16}"
            let ibanMatcher = NSPredicate(format: "SELF MATCHES %@", ibanPattern)
            if !ibanMatcher.evaluate(with: iban) {
                ibanError.text = "error_local_transaction_iban_invalid".localized
                return
            }
        }
        
        // Get amount
        
        if !amount.isEmpty {
            let amountPattern = "[0-9]{1,16}"
            let amountMatcher = NSPredicate(format: "SELF MATCHES %@", amountPattern)
            if !amountMatcher.evaluate(with: amount) {
                amountError.text = "error_local_transaction_amount_invalid".localized
                return
            }
        }

        let user = OrchestrationUser(identifier: identifier, domain: nil)
        let dataFields = [beneficiary, iban, amount]
        let crypoAppIndex = CryptoAppIndex.app4
        let protectionType = getSelectedProtectionType()
        let parameters = LocalTransactionParameters(user: user,
                                                    cryptoAppIndex: crypoAppIndex,
                                                    dataFields: dataFields,
                                                    protectionType: protectionType,
                                                    delegate: self)

        progressDialog = UIUtils.displayProgress(controller: self, message: "dialog_progress_local_transaction".localized)
        orchestrationDelegate.progressDialog = progressDialog
        dismissKeyboard()
        orchestrator?.startLocalTransaction(with: parameters)
    }
    
    // MARK: Protection selection
    private func getSelectedProtectionType() -> ProtectionType {
        if passwordButton.isSelected { return .password }
        if biometricButton.isSelected { return .biometric }
        return .noPassword
    }
}

extension LocalTransactionViewController: UITextFieldDelegate {
    func textFieldDidBeginEditing(_ textField: UITextField) {
        responder = textField
    }
    
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        // remove focus from current textfield
        textField.resignFirstResponder()
        
        if textField == beneficiary {
            iban.becomeFirstResponder()
        } else if textField == iban {
            amount.becomeFirstResponder()
        }
        
        return true
    }
    
    func textField(_ textField: UITextField, shouldChangeCharactersIn range: NSRange, replacementString string: String) -> Bool {
        let currentText = textField.text ?? ""
        guard let stringRange = Range(range, in: currentText) else { return false }
        let updatedText = currentText.replacingCharacters(in: stringRange, with: string)
        return updatedText.count <= Constants.Transactions.maxSizeForFieldInTransaction
    }
}

extension LocalTransactionViewController: LocalTransactionDelegate {
    func orchestrator(_ orchestrator: Orchestrator, didFinishLocalTransactionSuccessfullyWith signature: String, hostCode: String?) {
        UIUtils.hideProgress(progressDialog)
        UIUtils.displayAlert(controller: self,
                             title: "dialog_title_local_transaction".localized,
                             message: String(format: "dialog_content_local_transaction_success".localized,
                                             signature))
    }
    
    func orchestratorDidAbortLocalTransaction(_ orchestrator: Orchestrator) {
        UIUtils.hideProgress(progressDialog)
        UIUtils.displayAlert(controller: self,
                             title: "dialog_title_local_transaction".localized,
                             message: "dialog_content_local_transaction_abortion".localized)
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didReceiveLocalTransactionError error: OrchestrationError) {
        UIUtils.hideProgress(progressDialog)
        UIUtils.displayAlert(for: error, on: self)
    }
}

// MARK: Keyboard event
extension LocalTransactionViewController {
    @objc func keyboardWillShow(_ notification: Notification) {
        if let keyboardFrame = notification.userInfo?[UIResponder.keyboardFrameEndUserInfoKey] as? NSValue {
            let keyboardRectangle = keyboardFrame.cgRectValue
            let keyboardSize = keyboardRectangle.size
            let contentInsets = UIEdgeInsets(top: 0.0, left: 0.0, bottom: keyboardSize.height, right: 0.0)
            scroll.scrollIndicatorInsets = contentInsets
            scroll.contentInset = contentInsets
            
            var visibleFrame = scroll.frame
            visibleFrame.size.height -= keyboardSize.height
            if let responder = responder {
                let cellRect = scroll.convert(responder.frame, from: responder.superview)
                if !visibleFrame.contains(cellRect) {
                    scroll.scrollRectToVisible(cellRect, animated: true)
                }
            }
        }
    }

    @objc func keyboardWillHide(notification: NSNotification) {
        scroll.scrollsToTop = true
        scroll.contentInset = UIEdgeInsets.zero
        scroll.scrollIndicatorInsets = UIEdgeInsets.zero
    }
    
    @objc func dismissKeyboard() {
        responder?.resignFirstResponder()
        responder = nil
    }
}
